//
//  TwitchKit.m
//  Twitch Plugin
//
//  Created by Auston Stewart on 1/21/14.
//  Copyright (c) 2014 Justin.tv, Inc. All rights reserved.
//

#import "TwitchKit.h"
#import "TWAPICollectionViewController.h"
#import "TWLiveChannelsCollectionViewController.h"
#import "TWListNavigationController.h"
#import "TWAnalyticsController.h"
#import "TWWebViewController.h"
#import "TWLoginViewController.h"

NSString *const TKChannelListingWasPresentedNotification = @"TKChannelListingWasPresentedNotification";
NSString *const TKChannelListingWasDismissedNotification = @"TKChannelListingWasDismissedNotification";
NSString *const TKUserSelectedChannelNotification = @"TKUserSelectedChannelNotification";
NSString *const TKLoginViewWasDismissedNotification = @"TKLoginViewWasDismissedNotification";
NSString *const TKLoginViewWasPresentedNotification = @"TKLoginViewWasPresentedNotification";
NSString *const TKWillLaunchTwitchAppNotification = @"TKWillLaunchTwitchAppNotification";

NSString *const TKAuthTokenUserInfoKey = @"TKAuthTokenUserInfoKey";
NSString *const TKErrorUserInfoKey = @"TKErrorUserInfoKey";

@implementation TwitchKit

+ (TwitchKit *) sharedPlugin {
	static TwitchKit *sharedPlugin = nil;
	
	tw_dispatch_once(^{
		sharedPlugin = [[[self class] alloc] init];
	});
	
	return sharedPlugin;
}

- (id)init
{
	if ((self = [super init])) {
	
		// Register defaults
		[[NSUserDefaults standardUserDefaults] registerDefaults:@{TWLiveChannelListPresentationDefaultsKey:@(TWAPITableViewStyleMixed)}];
	}
	
	return self;
}

- (void)presentStreamListForGameTitle:(NSString *)gameTitle mode:(TKStreamPresentationMode)mode
{
	if (![UIDevice currentDevice].isRunningOS7) {
		
		NSLog(@"TwitchKit: Only devices running iOS 7.0 and later may use this function.");
		return;
	}
	
	// Instantiate the live channel collection view controller
	TWLiveChannelsCollectionViewController *streamsController = [[TWLiveChannelsCollectionViewController alloc] initWithGameTitle:gameTitle];
	streamsController.preferredStreamPresentationMode = mode;
	
	// Instantiate the encapsulating navigation controller
	TWListNavigationController *listNavController = [[TWListNavigationController alloc] initWithRootViewController:streamsController];
		
	[[UIApplication sharedApplication].keyWindow.rootViewController presentViewController:listNavController animated:YES completion:^{
		
		// Post a notification
		[[NSNotificationCenter defaultCenter] postNotificationName:TKChannelListingWasPresentedNotification object:self];
	}];
	
	// Log event to MixPanel
	[[TWAnalyticsController analyticsController] trackEvent:TWAnalyticsEventStreamsBrowse withDetails:@{TWAnalyticsPropertyGame:gameTitle}];
}
	
- (void)presentStreamForChannelNamed:(NSString *)channel mode:(TKStreamPresentationMode)mode
{
	if (!channel.length) return;
	
	if (![UIDevice currentDevice].isRunningOS7) {
		
		NSLog(@"TwitchKit: Only devices running iOS 7.0 and later may use this function.");
		return;
	}
	
	NSURL *appStreamURL = [NSURL URLWithString:[NSString stringWithFormat:@"twitch://stream/%@",channel]];
	
	if (mode == TKStreamPresentationModeUseTwitchAppIfInstalled &&
		[[UIApplication sharedApplication] canOpenURL:appStreamURL]) {
		
		// Send notification
		[[NSNotificationCenter defaultCenter] postNotificationName:TKWillLaunchTwitchAppNotification object:self userInfo:@{TWAnalyticsPropertyChannel:channel}];
		
		// Launch the Twitch app to view the stream
		[[UIApplication sharedApplication] openURL:appStreamURL];
	}
	else {
	
		// Launch the stream in a modal webview
		UINavigationController *webViewNavigationController = [UINavigationController webNavigationControllerWithURL:[NSURL URLWithString:[NSString stringWithFormat:@"http://www.twitch.tv/%@/embed",channel]]];
		TWWebViewController *webViewController = (TWWebViewController *)webViewNavigationController.topViewController;
		webViewController.webView.scalesPageToFit = YES;
		webViewNavigationController.modalPresentationStyle = UIModalPresentationFullScreen;
		webViewNavigationController.topViewController.title = channel;
		
		UIViewController *rootViewController = [UIApplication sharedApplication].keyWindow.rootViewController;
		[(rootViewController.presentedViewController ? rootViewController.presentedViewController : rootViewController) presentViewController:webViewNavigationController animated:YES completion:NULL];
	}
}

- (void)presentLoginViewForClientID:(NSString *)clientID redirectURI:(NSString *)redirectURI
{
	if (!clientID.length) return;
	
	if (![UIDevice currentDevice].isRunningOS7) {
		
		NSLog(@"TwitchKit: Only devices running iOS 7.0 and later may use this function.");
		return;
	}
	
	UINavigationController *loginNavigationController = [UINavigationController loginNavigationControllerForClientID:clientID redirectURI:redirectURI mode:TWLoginViewControllerModeLogin];
	if (![UIDevice currentDevice].isPhone)
		loginNavigationController.modalPresentationStyle = UIModalPresentationFormSheet;
	
	[[UIApplication sharedApplication].keyWindow.rootViewController presentViewController:loginNavigationController animated:YES completion:^{
		
		// Post a notification
		[[NSNotificationCenter defaultCenter] postNotificationName:TKLoginViewWasPresentedNotification object:self];
	}];
}

@end
